/*
 * Copyright (C) 2014 Yahoo Inc.
 */

// A constant prefix which is part of every attachment id
var ATTACHMENT_ID_PREFIX = "ymail_attachmentId";

// The class name for storing the content id of an attachment
var INLINE_ATTACHMENT_CONTENT_ID_CLASS_NAME_PREFIX = "inlined-image-cid-";

// A place holder token representing the attachment image data or the image url
var ATTACHMENT_ID_TOKEN = "#ID#";
var ATTACHMENT_URL_TOKEN = "#DATA_URL#";
var ATTACHMENT_PLACEHOLDER_URL_TOKEN = "#PLACEHOLDER_URL#";
var ATTACHMENT_CONTENT_ID_TOKEN = "#CONTENT_ID#";

// Various style class names used to style and look up elements
// These might be used in other files, make sure any name changes are reflected elsewhere
var INLINED_ATTACHMENT_CLASS_NAME = "inline-image-global";
var ATTACHMENT_IMAGE_ELEMENT_CLASS = "attachment-image";
var ATTACHMENT_PLACEHOLDER_ELEMENT_CLASS = "attachment-placeholder";

// A place holder token representing the attachment name
var ATTACHMENT_NAME_TOKEN = "#ATT_NAME#";

// A place holder token representing the attachment size
var ATTACHMENT_SIZE_TOKEN = "#ATT_SIZE#";

// Each attachment is represented by this template
var ATTACHMENT_TEMPLATE = "<img src=\"https://com.yahoo.mobile.client.android.mail.resource/drawable/mailsdk_compose_collapse_inline\" class=\"attachment-options-btn\">" +
        "<div class=\"" + ATTACHMENT_PLACEHOLDER_ELEMENT_CLASS + "\"><div class=\"center\"></div><img src=\"" + ATTACHMENT_PLACEHOLDER_URL_TOKEN + "\"/></div>" +
        "<div class=\"" + ATTACHMENT_IMAGE_ELEMENT_CLASS + "\" style=\"background-image: url(\'" + ATTACHMENT_URL_TOKEN +"\');\"></div>" +
        "<div class=\"attachment-name-label default-text-style\">" + ATTACHMENT_NAME_TOKEN + "</div>" +
        "<div class=\"attachment-size-label default-text-style\">" + ATTACHMENT_SIZE_TOKEN + "</div>";

/**
 * An object which controls the attachment band, including adding attachments into view and showing/hiding the detailed view.
 */
var attachmentsController = {
    attachmentsContainerNode: null,
    attachmentsNode: null,
    attachmentsLabelNode: null,
    composeContentNode: null,
    numAttachments: 0,
    totalSizeBytes: 0,

    /**
     * A method which starts the attachmentsController
     */
    init: function(attachmentsContainerNode, attachmentsNode, attachmentsLabelNode, composeContentNode) {
        this.attachmentsContainerNode = attachmentsContainerNode;
        this.attachmentsNode = attachmentsNode;
        this.attachmentsLabelNode = attachmentsLabelNode;
        this.composeContentNode = composeContentNode;
    },

    /**
     * Add an attachment with the given parameters. Usually called by native code.
     */
    addAttachment: function(id, name, size, imagePlaceHolder, imageUri, sizeString) {
        var attachmentNode = document.getElementById(ATTACHMENT_ID_PREFIX + id);

        if (!attachmentNode) {
            // New attachment, so create a node and configure it
            attachmentNode = document.createElement("li");
            attachmentNode.setAttribute("class", "attachment");
            attachmentNode.setAttribute("id", ATTACHMENT_ID_PREFIX + id);
            attachmentNode.addEventListener("click", this.attachmentClickHandler.bind(this), false);
            this.numAttachments++;
        } else {
            this.totalSizeBytes -= parseInt(attachmentNode.getAttribute("attachment-size"), 10);
        }

        attachmentNode.innerHTML = ATTACHMENT_TEMPLATE
                .replace(ATTACHMENT_URL_TOKEN, imageUri)
                .replace(ATTACHMENT_NAME_TOKEN, name)
                .replace(ATTACHMENT_SIZE_TOKEN, sizeString)
                .replace(ATTACHMENT_PLACEHOLDER_URL_TOKEN, imagePlaceHolder);

        var image = new Image();
        image.addEventListener("load", this.attachmentImageLoadHandler.bind(this, attachmentNode), true);
        image.setAttribute("src", imageUri);

        if (this.numAttachments > 0) {
            this.attachmentsContainerNode.classList.remove("hidden");
        }

        // Update the size
        this.totalSizeBytes += size;

        attachmentNode.setAttribute("attachment-size", size);

        this.attachmentsNode.appendChild(attachmentNode);

        this.attachmentsLabelNode.textContent = yMailBridge.getAttachmentBandString(this.numAttachments, this.totalSizeBytes);

        attachmentNode.lastChild.setAttribute("aria-label", name);
    },

    /**
     * Adds an inline attachment at the current cursor position
     */
    addInlineAttachment: function(id, imagePlaceHolder, imageUri, contentId) {
        // Prevent browser from aggressively caching modified inline image
        // Query param will be removed before sending
        var delimiter = imageUri.indexOf("?") > 0 ? "&" : "?",
            noCacheUri = imageUri + delimiter + NO_CACHE_QUERY_PARAM,
            imageId = ATTACHMENT_ID_PREFIX + id,
            containerNode = document.createElement('div'),
            imgNode = document.createElement('img'),
            imgController = new ImageViewController();

        // Make sure the focus is in the compose message body
        composeContentFocusController.focusContentNode();

        // Insert the HTML Template representing an inline attachment
        containerNode.className = UNIFIED_IMG_EL_CLASS_NAME + " " + INLINED_ATTACHMENT_CONTAINER_CLASS_NAME + " " + CARD_EL_CLASS_NAME;
        imgNode.id = imageId;
        imgNode.className = INLINED_ATTACHMENT_CLASS_NAME + " " + INLINE_ATTACHMENT_CONTENT_ID_CLASS_NAME_PREFIX + contentId;
        imgNode.setAttribute("src", noCacheUri);
        imgNode.setAttribute("style", "width:100%;max-width:800px;");
        containerNode.appendChild(imgNode);
        containerNode.appendChild(document.createElement('br'));

        // Attach the image controller to the new image
        containerNode = imgNode.parentNode;
        containerNode.contentEditable = "false";

        imgController.attach(imgNode);
        imgController.renderImage(imgNode);
    },

    /**
     * Adds a Cloud Card at the current selection.
     */
    addCloudAttachment: function(url, thumbnailURL, title, subtitle) {
        // Make sure the focus is in the compose message body
        composeContentFocusController.focusContentNode();

        // Create and insert the new Cloud Card
        AttachmentCardFactory.createCloudCard(url, thumbnailURL, title, subtitle).renderIntoSelection(true);

        // Call the auto save controller to immediately save the message body with the new card
        autoSaveController.handleInput(null);
    },

    /**
     * Add Gif Card.
     *
     * @param {string} gifUrl - source url for the gif image
     * @param {string} authorName - author's name used for attribution
     * @param {string} authorPageUrl - author's webpage used for attribution
     * @param {string} platformLogoSrcUrl - source url for the platform (example: Tumblr), used for attribution
     * @param {boolean} appendAttribution - true if we want to appendAttribution
     * @param {boolean} isTenorGif - true if it is from Tenor, Tumblr if false
     */
    addGifCard: function(gifUrl, authorName, authorPageUrl, platformLogoSrcUrl, appendAttribution, isTenorGif) {
        // Make sure the focus is in the compose message body
        composeContentFocusController.focusContentNode();

        // Create and insert the new GIF Card
        AttachmentCardFactory.createGifCard(gifUrl, authorName, authorPageUrl, platformLogoSrcUrl, appendAttribution, isTenorGif).renderIntoSelection(true);

        // Call the auto save controller to immediately save the message body with the new card
        autoSaveController.handleInput(null);
    },

    /**
     * Initialize any existing card attachments.
     * Including image card(img and gif) and enhancer card(cloud card and link enhancer card)
     *
     * @param {HtmlElement} containerEl container element used to search for existing Attachment Cards
     */
    initAttachmentCards: function(containerEl) {

        utils.forEach(containerEl.querySelectorAll('a.' + CLASS_LINK_ENHANCR_LINK), function(linkEl) {
            //If wrapped under 'PRE' tag, we don't want to enhance
            if (!utils.isDescendantOfTag('PRE', linkEl)) {
                linkEl.classList.remove(CLASS_LINK_ENHANCR_LINK);
                LinkEnhancr.fetchAndSetLinkEnhancr(linkEl.href, linkEl.id);
            }
        });

        // Query the DOM for unified classname as well as android only class name to locate card element
        // Note: Open draft with card and save on other platform will add yiv prefix to the existing classname that invalid query by exact classname,
        // so use partial match query to locate the card
        utils.forEach(containerEl.querySelectorAll("div[class*=" + UNIFIED_IMG_EL_CLASS_NAME + "]," +
                // Gif card from android client
                "[class*=" + GIF_CARD_EL_CLASS_NAME + "], " +
                // Gif/cloud card from iOS client
                "div." + IOS_LINK_CARD_CLASS_NAME + ", " +
                // Unified card class from other platform
                "[class*=" + UNIFIED_CARD_EL_CLASS_NAME + "]"), function(el) {

            var removeBtnEl = el.querySelector("button[class*=" + REMOVE_BTN_CLASS_NAME + "]"),
                // Attach remove button to gif card and attach "more options" button to inline img
                gifCard = el.querySelector('a img[src$=".gif"]'),
                imgContainer;


            var linkEnhancrCard = el.querySelector('table.card.yahoo-ignore-table');
            //var linkPreviewMainElement =  el.querySelector('.yahoo-link-enhancr-card'); el.classList.contains('.yahoo-link-enhancr-card');
            //We don't want to attach 'x' button, but we want to add "...(more)" button
            if (linkEnhancrCard) {
                el.style.position = "relative";
                el.setAttribute('contentEditable', false);
                var moreButton = el.querySelector("button[class*=" + CLASS_LINK_ENHANCR_MORE_BUTTON + "]");
                if (moreButton) {
                    moreButton.addEventListener('click', LinkEnhancr.handleMoreButtonOnClick, false);
                } else {
                      moreButton = LinkEnhancr.getMoreButton(el.id);
                      el.appendChild(moreButton);
                }
                return;
            }
            if (!removeBtnEl) {
                if (gifCard) {
                    removeBtnEl = CardViewController.prototype.createRemoveBtnEl();
                    // Desktop has a expand/shrink function for gif/img card and set the width to less than 100% that we don't support
                    // Always show the gif with original size
                    gifCard.style.width = "100%";
                    imgContainer = gifCard.parentNode;
                    // Attach remove button and set style for the container
                    imgContainer.insertBefore(removeBtnEl, gifCard);
                    imgContainer.parentNode.style.position = "relative";
                    imgContainer.parentNode.style.display = "inline-block";
                    el.setAttribute('contentEditable', false);
                } else if (el.className && (el.className.indexOf(UNIFIED_CARD_EL_CLASS_NAME) >= 0 ||  el.className.indexOf(IOS_LINK_CARD_CLASS_NAME) >= 0)) {
                    removeBtnEl = CardViewController.prototype.createRemoveBtnEl();
                    el.appendChild(removeBtnEl);
                    el.style.position = "relative";
                    el.setAttribute('contentEditable', false);
                }
            } else {
                // In case of yiv prefix added to remove button class, re-add the class name
                removeBtnEl.classList && removeBtnEl.classList.add(REMOVE_BTN_CLASS_NAME);
            }

            if (el && removeBtnEl) {
                (new CardViewController()).attach(el, removeBtnEl);
            }
        });
    },

    /**
     * Remove an attachment with the given id
     */
    removeAttachment: function(id) {
        var attachmentNode = document.getElementById(ATTACHMENT_ID_PREFIX + id);

        if (attachmentNode) {
            attachmentNode.parentNode.removeChild(attachmentNode);

            this.totalSizeBytes -= parseInt(attachmentNode.getAttribute("attachment-size"), 10);

            this.numAttachments--;

            this.attachmentsLabelNode.textContent = yMailBridge.getAttachmentBandString(this.numAttachments, this.totalSizeBytes);

            if (this.numAttachments === 0) {
                this.attachmentsContainerNode.classList.add("hidden");
            }
        }
    },

    /**
     * Click handler for each attachment
     */
    attachmentClickHandler: function(event) {
        yMailBridge.showAttachmentClickOptions(event.currentTarget.id.replace(ATTACHMENT_ID_PREFIX, ""));
    },

    /**
     * This method is used to handle errors while loading attachment preview images
     */
    attachmentImageLoadHandler: function(attachmentNode) {
        attachmentNode.getElementsByClassName(ATTACHMENT_IMAGE_ELEMENT_CLASS)[0].style.visibility = "visible";
        attachmentNode.getElementsByClassName(ATTACHMENT_PLACEHOLDER_ELEMENT_CLASS)[0].style.visibility = "hidden";
    }
};
